c==========================================================================
c--------------------------------------------------------------------------
c==========================================================================
      module FIBER_1D_model_module
c==========================================================================
      integer*4, parameter :: FIBER_1D_ELAS_E    = 1,
     &                        FIBER_1D_ELAS_v    = 2,
     &                        FIBER_1D_ELAS_MAX  = 2

      integer*4, parameter :: FIBER_1D_MAIN_N_TENSILE_CURVE_PTS = 1,
     &                        FIBER_1D_MAIN_N_COMPRES_CURVE_PTS = 2,
     &                        FIBER_1D_MAIN_CHAR_LENGTH         = 3,
     &                        FIBER_1D_MAIN_COUPLED_TENS_COMPR_SOFT = 4,
     &                        FIBER_1D_MAIN_E0_CHOICE           = 5

      
      integer*4, parameter :: FIBER_1D_NONL_N_TENSILE_CURVE_PTS =  1,
     &                        FIBER_1D_NONL_N_COMPRES_CURVE_PTS =  2,
     &                        FIBER_1D_NONL_TENSILE_LIST_ADR    =  3,
     &                        FIBER_1D_NONL_COMPRES_LIST_ADR    = 23

      integer*4, parameter :: FIBER_1D_MAX_PTS = 100 ! do not increase without updating size in models.for

      type MatProperties_FIBER_1D
        ! input data
        real*8    :: E,v
        real*8    :: Lc ! characteristic length
        integer*4 :: n_T_pts
        integer*4 :: n_C_pts
        real*8    :: T_curve (2,FIBER_1D_MAX_PTS)
        real*8    :: C_curve (2,FIBER_1D_MAX_PTS)
        real*8    :: ET_o,EC_o
        integer*4 :: Coupled_compr_tens_softening
        integer*4 :: Compr_soft_is_possible
        real*8    :: Compr_soft_eps
      end type MatProperties_FIBER_1D


      type StateParameters_FIBER_1D
        real*8 :: Stress 
        real*8 :: Stress0
        real*8 :: Strain
        real*8 :: Max_T_Strain
        real*8 :: Max_C_Strain
      end type StateParameters_FIBER_1D
      integer*4, parameter :: StateParameters_FIBER_1D_SIZE_INT4 =5 * 2


      character*8 ElasticProps_FIBER_1D (FIBER_1D_ELAS_MAX)
      data ElasticProps_FIBER_1D / "E    =  ",
     &                             "v    =  "/

      !xxxxxxxx
      !--------
      contains
      !--------
      !xxxxxxxx


      ! ===========================================================================
      integer*4 function FIBER_1D_StateStorageSizeI4words ()
      ! ===========================================================================
      include '..\src\nodecl.inc'

      ! storage size for stateN and Nx1 (actual stress dimension is taken into account)
      FIBER_1D_StateStorageSizeI4words = 
     &                    StateParameters_FIBER_1D_SIZE_INT4 * 2

      end function FIBER_1D_StateStorageSizeI4words


      ! ===========================================================================
      subroutine FIBER_1D_unpack_props (props,Mat)
      ! ===========================================================================
      include '..\src\nodecl.inc'
      include '..\src\prop.inc'
      include '..\src\math.inc'

      real*8, intent (in) :: props(*) 
      type (MatProperties_FIBER_1D), intent (out) :: Mat
      integer*4 :: offs,i
      real*8 :: aux,sig_max,sig_tol

      real*8 :: PRO_GetFromDAT
      
      Mat % E = PRO_GetFromDAT ( props,IPRO_GR_ELAS,FIBER_1D_ELAS_E)
      Mat % v = PRO_GetFromDAT ( props,IPRO_GR_ELAS,FIBER_1D_ELAS_v)

      Mat % n_T_pts = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                          FIBER_1D_NONL_N_TENSILE_CURVE_PTS)
      Mat % n_C_pts = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                          FIBER_1D_NONL_N_COMPRES_CURVE_PTS)

      if ( Mat % n_T_pts.gt.FIBER_1D_MAX_PTS ) then
        call sys_stop ("too many points declared for tension branch of c
     &oncrete model - max=100 pts")
      end if
      if ( Mat % n_C_pts.gt.FIBER_1D_MAX_PTS ) then
        call sys_stop ("too many points declared for compession branch o
     &f concrete model - max=100 pts")
      end if

      offs = 0
      do i=1,Mat % n_T_pts
        Mat % T_curve (1,i) = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                         FIBER_1D_NONL_TENSILE_LIST_ADR+offs+0)
        Mat % T_curve (2,i) = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                         FIBER_1D_NONL_TENSILE_LIST_ADR+offs+1)
        offs=offs+2
      end do

      offs = 0
      do i=1,Mat % n_C_pts
        Mat % C_curve (1,i) = -PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                         FIBER_1D_NONL_COMPRES_LIST_ADR+offs+0)
        Mat % C_curve (2,i) = -PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                         FIBER_1D_NONL_COMPRES_LIST_ADR+offs+1)
        offs=offs+2
      end do

      Mat % ET_o = 0.0d0
      if (Mat % n_T_pts.gt.0 ) then
        Mat % ET_o = (Mat % T_curve (2,2)-Mat % T_curve (2,1))/
     &               (Mat % T_curve (1,2)-Mat % T_curve (1,1))  
      end if

      Mat % EC_o = 0.0d0
      if (Mat % n_C_pts.gt.0 ) then
        Mat % EC_o = (Mat % C_curve (2,2)-Mat % C_curve (2,1))/
     &               (Mat % C_curve (1,2)-Mat % C_curve (1,1))  
      end if

      Mat % Lc = PRO_GetFromDAT ( props,IPRO_GR_MAIN,
     &                                  FIBER_1D_MAIN_CHAR_LENGTH)
      
      Mat % Coupled_compr_tens_softening = 
     &                  PRO_GetFromDAT ( props,IPRO_GR_MAIN,
     &                      FIBER_1D_MAIN_COUPLED_TENS_COMPR_SOFT)

      Mat % Compr_soft_is_possible = 0
      Mat % Compr_soft_eps         = 0.0d0

      sig_max = 0.0d0
      do i=1,Mat % n_C_pts
        sig_max = dmax1 ( sig_max,dabs(Mat % C_curve (2,i)))
      end do
      sig_tol = 1.0d-6 * sig_max

      if ( Mat % Coupled_compr_tens_softening.gt.0 ) then
        do i=2,Mat % n_C_pts
          if ( Mat % C_curve (2,i)+sig_tol.lt.Mat % C_curve (2,i-1))then
            Mat % Compr_soft_is_possible = 1
            Mat % Compr_soft_eps = Mat % C_curve (1,i) 
          end if
        end do
      end if
      
      end subroutine FIBER_1D_unpack_props



      ! ===========================================================================
      subroutine FIBER_1D_repack_state (buff,offs,state,CopyProc)
      ! ===========================================================================
      use LPTR_profiler
      include '..\src\nodecl.inc'
      integer*4                                  :: buff (*)
      integer*4,                  intent (in )   :: offs
      type (StateParameters_FIBER_1D)            :: state
      external CopyProc

      integer*4 :: k
      type (LPTR_PtrsOrganizer) :: myPtrs

      call LPTR_newWithOffs ( myPtrs,offs )
      
      k = LPTR_next   (myPtrs,2 ) 
      call CopyProc   ( state % Stress ,buff (k),2 )
      k = LPTR_next   (myPtrs,2 ) 
      call CopyProc   ( state % Stress0,buff (k),2 )
      k = LPTR_next   (myPtrs,2 ) 
      call CopyProc   ( state % Strain ,buff (k),2 )
      k = LPTR_next   (myPtrs,2 ) 
      call CopyProc   ( state % Max_T_Strain ,buff (k),2 )
      k = LPTR_next   (myPtrs,2 ) 
      call CopyProc   ( state % Max_C_Strain ,buff (k),2 )

      end subroutine FIBER_1D_repack_state


c==========================================================================
      subroutine FIBER_1D_InitStateParameters (StateN,StateNx1,Mat)
c==========================================================================
      include '..\src\nodecl.inc'
      type (StateParameters_FIBER_1D):: StateN,StateNx1      
      type (MatProperties_FIBER_1D)  :: Mat

      StateNx1 % Stress       = 0.0d0
      StateNx1 % Stress0      = 0.0d0
      StateNx1 % Strain       = 0.0d0
      StateNx1 % Max_T_Strain = 0.0d0
      StateNx1 % Max_C_Strain = 0.0d0

      end subroutine FIBER_1D_InitStateParameters




c==========================================================================
      subroutine FIBER_1D_NewState ( dE,dSig0,StateN,StateNx1,Es,Et,Mat)
c==========================================================================
      include '..\src\nodecl.inc'
      type (StateParameters_FIBER_1D):: StateN,StateNx1
      real*8                         :: dE,dSig0,E,Eep
      type (MatProperties_FIBER_1D)  :: Mat
      real*8                         :: Es,Et,Stress,loading_vector
      real*8                         :: StressTMP,Etx,Eo
      real*8, parameter              :: RESID_ET_MULT = 1.0d-4
      
      StateNx1 % Strain = StateN % Strain + dE

      Eo = dmax1 ( Mat % ET_o, Mat % EC_o )

      if ( dabs(dE).le.1.0d-15 ) then
        StateNx1 = StateN
        Es = Eo
        Et = Es
        return        
      end if

      ! singular case
      if ( StateN % Max_T_Strain.eq.0.0d0.and.
     &     StateN % Max_C_Strain.eq.0.0d0 ) then
        if ( dabs(StateNx1 % Strain).le.1.0d-15 ) then
          Stress = 0.0d0
          Es = Eo
          Et = Es
        else
          call FIBER_1D_getStressFromCurve (StateNx1 % Strain,Mat,
     &                                      StateNx1 % Stress,Etx)
          if ( StateNx1 % Strain.gt.0.0d0 ) then
            StateNx1 % Max_T_Strain = StateNx1 % Strain 
            Es = StateNx1 % Stress / StateNx1 % Max_T_Strain
            Et = Etx          
            Et = dmax1 ( Et,RESID_ET_MULT*Eo )
          else
            StateNx1 % Max_C_Strain = StateNx1 % Strain 
            Es = StateNx1 % Stress / StateNx1 % Max_C_Strain
            Et = Etx         
            Et = dmax1 ( Et,RESID_ET_MULT*Eo )
          end if
        end if
        return
      end if

      loading_vector = dsign (1.0d0,StateN % Strain)
      if ( loading_vector * dE.ge.0.0d0 ) then

          !--------------------
          ! continue reloading 
          !--------------------
         select case ( StateN % Strain.gt.0.0d0)

            !--------------------
            case ( .true. )
            !--------------------
              if ( StateNx1 % Strain.gt.StateN % Max_T_Strain ) then
                ! again virgin loading
                call FIBER_1D_getStressFromCurve (
     &               StateNx1 % Strain,Mat,StateNx1 % Stress,Etx)
                StateNx1 % Max_T_Strain = StateNx1 % Strain 
                Es = StateNx1 % Stress / StateNx1 % Max_T_Strain
                Et = Es
                if ( dabs(dE).gt.1.0d-12 ) then
cc                Et = (StateNx1%Stress-StateN%Stress)/dE
                  Et = Etx
c                 Et = dmax1 ( Et,RESID_ET_MULT*Eo )
                end if
              else
                ! still reloading and not yet virgin loading
                call FIBER_1D_getStressFromCurve (
     &               StateNx1 % Max_T_Strain,Mat,StressTMP,Etx)                
                Es = StressTMP / StateN % Max_T_Strain
                Et = Es
                StateNx1 % Stress = Es * StateNx1 % Strain
              end if

            !--------------------
            case (.false.)
            !--------------------

              if ( StateNx1 % Strain.le.StateN % Max_C_Strain ) then
                ! again virgin loading
                call FIBER_1D_getStressFromCurve (
     &                     StateNx1 % Strain,Mat,StateNx1 % Stress,Etx)
                StateNx1 % Max_C_Strain = StateNx1 % Strain 
                Es = StateNx1 % Stress / StateNx1 % Max_C_Strain
                Et = Es
                if ( dabs(dE).gt.1.0d-12 ) then
c                 Et = (StateNx1%Stress-StateN%Stress)/dE
                  Et = Etx
c                 Et = dmax1 ( Et,RESID_ET_MULT*Eo )
                end if
              else
                ! still reloading and not yet virgin loading
                call FIBER_1D_getStressFromCurve (
     &                        StateNx1 % Max_C_Strain,Mat,StressTMP,Etx)                
                Es = StressTMP / StateN % Max_C_Strain
                Et = Es
                StateNx1 % Stress = Es * StateNx1 % Strain
              end if

          end select
       

      else

        !--------------------
        ! unloading process 
        !--------------------
c       select case ( StateN % Stress.gt.0.0d0)
        select case ( StateN % Strain.gt.0.0d0)

          !--------------------
          case ( .true. )
          !--------------------
            ! unloading from tension
            if ( StateNx1 % Strain.ge.0.0d0 ) then
              ! still tensile stress
              call FIBER_1D_getStressFromCurve (
     &                   StateNx1 % Max_T_Strain,Mat,StressTMP,Etx)                
              Es = StressTMP / StateN % Max_T_Strain
              StateNx1 % Stress = Es * StateNx1 % Strain
              Et = Es
            else
              ! switch to compression
              if ( StateNx1 % Strain.lt.StateN % Max_C_Strain ) then          

                ! to virgin compression
                call FIBER_1D_getStressFromCurve (
     &                     StateNx1 % Strain,Mat,StateNx1 % Stress,Etx)                
                StateNx1 % Max_C_Strain = StateNx1 % Strain
                Es = StateNx1 % Stress / StateNx1 % Max_C_Strain
                Et = Es
                if ( dabs(dE).gt.1.0d-12 ) then
cc                Et = (StateNx1%Stress-StateN%Stress)/dE
c                 Et = dmax1 ( Et,RESID_ET_MULT*Eo )
                  Et=Etx
                end if

              else

                ! on reloading compression path
                call FIBER_1D_getStressFromCurve (
     &                     StateNx1 % Max_C_Strain,Mat,StressTMP,Etx)                
                Es = StressTMP / StateN % Max_C_Strain
                StateNx1 % Stress = Es * StateNx1 % Strain
                Et = Es
              end if
            end if

          !--------------------
          case (.false.)
          !--------------------
            ! unloading from compression
            if ( StateNx1 % Strain.lt.0.0d0 ) then
              ! still compression stress
              call FIBER_1D_getStressFromCurve (
     &             StateNx1 % Max_C_Strain,Mat,StressTMP,Etx)                
              Es = StressTMP / StateN % Max_C_Strain
              StateNx1 % Stress = Es * StateNx1 % Strain
              Et = Es
            else
              ! switch to tension
              if ( StateNx1 % Strain.gt.StateN % Max_T_Strain ) then          
                ! to virgin tension
                call FIBER_1D_getStressFromCurve (
     &               StateNx1 % Strain,Mat,StateNx1 % Stress,Etx)                
                StateNx1 % Max_T_Strain = StateNx1 % Strain 
                Es = StateNx1 % Stress / StateNx1 % Max_T_Strain
                Et = Es
                if ( dabs(dE).gt.1.0d-12 ) then
cc                  Et = (StateNx1%Stress-StateN%Stress)/dE
c                  Et = dmax1 ( Et,RESID_ET_MULT*Eo )
                  Et = Etx
                end if

              else

                ! on reloading tension path
                call FIBER_1D_getStressFromCurve (
     &               StateNx1 % Max_T_Strain,Mat,StressTMP,Etx)                
                Es = StressTMP / StateN % Max_T_Strain
                StateNx1 % Stress = Es * StateNx1 % Strain
                Et = Es
              end if
            end if
        end select
       
      end if

      Et = dmax1 ( Et,RESID_ET_MULT*Eo )

c     call deb_1 ( Et,1,1,'Et')

      end subroutine FIBER_1D_NewState


c==========================================================================
      subroutine FIBER_1D_InterpolateCurve ( value,curve,npts,Stress,Et) 
c==========================================================================
      include '..\src\nodecl.inc'
      integer*4 :: npts
      real*8    :: value,curve (2,*),Stress,Et
      integer*4 :: i 
      logical*4 :: takeNext
      real*8    :: eps

      if ( npts.le.0 ) then
        Stress = 0.0d0
        return
      end if
      
      i = 1
      takeNext = .true.
      do while( takeNext )
        eps = curve (1,i)
        takeNext = (dabs(value).gt.dabs(eps))
        if (takeNext) i=i+1
        takeNext = (i.le.npts.and.takeNext)
      end do

      if  ( i.eq.1 ) then                 ! before the first point
                                          ! take as at first
        Stress = curve (2,1)
        Et     = (curve (2,2)-curve(2,1))/(curve (1,2)-curve(1,1))

      else if ( i.gt.npts ) then

        Stress = curve (2,npts)
        Et = 0.0d0

      else                        ! interpolate between i-1 and i point

        Stress     = curve (2,i-1) + 
     &                     (curve (2,i)-curve(2,i-1))  /
     &                     (curve (1,i)-curve(1,i-1))  *  
     &                     (value-curve (1,i-1))       
        Et = (curve (2,i)-curve(2,i-1))/(curve (1,i)-curve(1,i-1))

      end if

      end subroutine FIBER_1D_InterpolateCurve


c==========================================================================
      subroutine FIBER_1D_getStressFromCurve ( Strain,Mat,Stress,Et )
c==========================================================================
      include '..\src\nodecl.inc'
      real*8 :: Strain,Stress,Et
      type (MatProperties_FIBER_1D)  :: Mat


      if ( Strain.ge.0.0d0 ) then
        if ( Mat % n_T_pts.le.0 ) then
          Stress = 0.0d0
        else
          if ( Strain.lt.0.0d0 ) stop "FIBER_1D_getStressFromCurve"
          call FIBER_1D_InterpolateCurve ( Strain,Mat % T_curve,
     &                                     Mat % n_T_pts,Stress,Et ) 
        end if
      else
        if ( Mat % n_C_pts.le.0 ) then
          Stress = 0.0d0
        else
          if ( Strain.gt.0.0d0 ) stop "FIBER_1D_getStressFromCurve"
          call FIBER_1D_InterpolateCurve ( Strain,Mat % C_curve,
     &                                     Mat % n_C_pts,Stress,Et ) 
        end if
      end if

      end subroutine FIBER_1D_getStressFromCurve


c==========================================================================
      function FIBER_1D_E_init ( Mat ) result (Eo)
c==========================================================================
      include '..\src\nodecl.inc'
      type (MatProperties_FIBER_1D)  :: Mat
      real*8 :: Eo

      Eo  = 0.5d0 * ( Mat % ET_o + Mat % EC_o )
     
      end function FIBER_1D_E_init



c==========================================================================
      function FIBER_1D_E_secant ( StateNx1,Mat ) result (Es)
c==========================================================================
      include '..\src\nodecl.inc'
      type (StateParameters_FIBER_1D):: StateNx1
      type (MatProperties_FIBER_1D)  :: Mat
      real*8 :: Es

      if ( StateNx1 % Strain.gt.0.0d0 ) then
        if ( StateNx1 % Strain.gt.1.0d-12 ) then
          Es = StateNx1 % Stress / StateNx1 % Max_T_Strain           
        else
          Es  = Mat % ET_o
        end if

      else if ( StateNx1 % Strain.lt.0.0d0 ) then

        if ( StateNx1 % Strain.lt.-1.0d-12 ) then
          Es = StateNx1 % Stress / StateNx1 % Max_T_Strain           
        else
          Es = Mat % EC_o
        end if

      else
    
        Es = dmax1 ( Mat % ET_o,Mat % EC_o )
        
      end if

     
      end function FIBER_1D_E_secant



c==========================================================================
c--------------------------------------------------------------------------
c==========================================================================
      end module FIBER_1D_model_module
c==========================================================================
c--------------------------------------------------------------------------
c==========================================================================


      ! ===========================================================================
      subroutine FIBER_1D_CopyLeftToRight (a,b,n)
      ! ===========================================================================
      include '..\src\nodecl.inc'
 
      integer*4 :: n,a(n),b(n)

      b = a

      end subroutine FIBER_1D_CopyLeftToRight

      ! ===========================================================================
      subroutine FIBER_1D_CopyRightToLeft (a,b,n)
      ! ===========================================================================
      include '..\src\nodecl.inc'
 
      integer*4 :: n,a(n),b(n)

      a = b

      end subroutine FIBER_1D_CopyRightToLeft



c========================================================================
      subroutine FIBER_1D_PrintProps( Props, InfoConst )
c------------------------------------------------------------------------
c calls functions to print parameters for all material data groups
c for nonactive groups use dummy PRO_DummyPrint   function

      include '..\src\nodecl.inc'
      include '..\src\prop.inc'
      include '..\src\propname.inc'

      real*8    :: Props     (*)
      integer*4 :: InfoConst (*)

  ! function to deal with each group of properties in this model
      external  PRO_MainPrint,
     &          PRO_DensPrint,
     &          FIBER_1D_ElasticPrintParam,
     &          PRO_DummyPrint,
     &          FLU_PrintParam,
     &          CRE_PrintParam,
     &          FIBER_1D_NonlPrintParam,
     &          HEA_PrintParam,
     &          HUM_PrintParam,
     &          PRO_InisPrint,
     &          PRO_LocStabPrint,
     &          PRO_DampPrint


       call PRO_PrintParam( Props, InfoConst ,
     &          'Concrete 1D model for beams',
     &          PRO_MainPrint,      ! function which prints parameters from group MAIN (standard)
     &          PRO_DensPrint,      ! function which prints parameters from group DENS (standard)
     &          FIBER_1D_ElasticPrintParam,! function which prints parameters from group ELAS
     &          PRO_DummyPrint,     ! dummy function which does nothing
     &          PRO_DummyPrint,     ! function which prints parameters for group FLOW
     &          PRO_DummyPrint,     ! function which prints parameters for group CREEP (standard)
     &          FIBER_1D_NonlPrintParam, ! function which prints parameters for group NONL
     &          HEA_PrintParam,     ! function which prints parameters for group HEAT
     &          HUM_PrintParam,     ! function which prints parameters for group HUMI (standard)
     &          PRO_DummyPrint,     ! function which prints parameters for group INIS (standard)
     &          PRO_LocStabPrint,   ! function which prints parameters for group LSTAB(standard)
     &          PRO_DampPrint,
     &          PRO_DummyPrint)

      return
      end

c==================================================================
      subroutine  FIBER_1D_ElasticPrintParam( ifile,props)
c------------------------------------------------------------------
c print elastic parameters
      use FIBER_1D_model_module
      include '..\src\nodecl.inc'
      include '..\src\prop.inc'
      include '..\src\prnline.inc'

      integer*4 :: ifile
      real*8    :: props(*)
      real*8    :: tmp
      real*8    :: PRO_GetFromDAT
      integer*4 :: i
      type (print_line_struct_type) :: line

      do i=1,FIBER_1D_ELAS_MAX
        call IO_BeginLine_ex (line)
        call IO_AddStrToLine_ex  ( line, ElasticProps_FIBER_1D (i),8 )
        tmp = PRO_GetFromDAT  ( props,IPRO_GR_ELAS,i )
        call IO_AddR8ToLine_ex   ( line, tmp )
        call IO_EndLine_ex       ( line, ifile ) 
      end do


      return
      end


c==================================================================
      subroutine  FIBER_1D_NonlPrintParam( ifile,props)
c------------------------------------------------------------------
c print nonlinear parameters
      use FIBER_1D_model_module
      include '..\src\nodecl.inc'
      include '..\src\prop.inc'
      include '..\src\prnline.inc'

      integer*4 :: ifile,i
      real*8    :: props(*),tmp
      real*8    :: PRO_GetFromDAT 
      type (MatProperties_FIBER_1D)  :: Mat
      type (print_line_struct_type) :: line

      call FIBER_1D_unpack_props (props,Mat)
   
      call IO_BeginLine_ex (line)
      call IO_AddStrToLine_ex  (line,
     &                "TENSILE BRANCH   -->  NR OF PTS  = ",35 )
      call IO_AddI4ToLine_ex   ( line,Mat % n_T_pts )
      call IO_EndLine_ex       ( line,ifile ) 
   

      do i=1,Mat % n_T_pts
        call IO_BeginLine_ex (line)
        call IO_AddI4ToLine_ex  ( line, i )
        call IO_AddStrToLine_ex ( line, " ",1 )
        call IO_AddR8ToLine_ex  ( line, Mat % T_Curve (1,i) )
        call IO_AddStrToLine_ex ( line, " ",1 )
        call IO_AddR8ToLine_ex  ( line, Mat % T_Curve (2,i) )
        call IO_EndLine_ex      ( line, ifile ) 
      end do


      call IO_BeginLine_ex (line)
      call IO_AddStrToLine_ex (line,
     &        "COMPRESSION BRANCH -> NR OF PTS  = ",35 )
      call IO_AddI4ToLine_ex   ( line, Mat % n_C_pts )
      call IO_EndLine_ex       ( line, ifile ) 

      do i=1,Mat % n_C_pts
        call IO_BeginLine_ex (line)
        call IO_AddI4ToLine_ex (line, i )
        call IO_AddStrToLine_ex (line, " ",1 )
        call IO_AddR8ToLine_ex  (line, Mat % C_Curve (1,i) )
        call IO_AddStrToLine_ex (line, " ",1 )
        call IO_AddR8ToLine_ex  (line, Mat % C_Curve (2,i) )
        call IO_EndLine_ex      (line, ifile ) 
      end do

      return
      end


c ===================================================================
      subroutine FIBER_1D_ScaleCurves (props,Coeff)
c ===================================================================
      use FIBER_1D_model_module
      include '..\src\nodecl.inc'
      include '..\src\prop.inc'
      real*8    :: props (*),Coeff
      integer*4 :: offs,i,n_T_pts,n_C_pts
      real*8    :: PRO_GetFromDAT
      real*8    :: value

      n_T_pts = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                           FIBER_1D_NONL_N_TENSILE_CURVE_PTS)
      n_C_pts = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                           FIBER_1D_NONL_N_COMPRES_CURVE_PTS)

      offs = 0
      do i=1,n_T_pts
        value = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                           FIBER_1D_NONL_TENSILE_LIST_ADR+offs+1)
        value = value * Coeff
        call PRO_PutParamNonl  ( props,FIBER_1D_NONL_TENSILE_LIST_ADR+
     &                           offs+1,value)
        offs = offs + 2
      end do

      offs = 0
      do i=1,n_C_pts
        value = PRO_GetFromDAT ( props,IPRO_GR_NONL,
     &                           FIBER_1D_NONL_COMPRES_LIST_ADR+offs+1)
        value = value * Coeff
        call PRO_PutParamNonl  ( props,FIBER_1D_NONL_COMPRES_LIST_ADR+
     &                           offs+1,value)
        offs = offs + 2
      end do

      end subroutine FIBER_1D_ScaleCurves


c ===================================================================
      subroutine FIBER_1D_ModifParam ( props,Coeff  )
c ===================================================================
c this routine makes modification of plastic parameters like c,phi,psi
c in case of stability analysis
c please make modifications only after ! HERE MAKE AN ACTION comments

      use LocalStabilityStorage
      use FIBER_1D_model_module

c!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
c  !MS$ ATTRIBUTES DLLIMPORT :: IPRO_GetStabilityTypeForMaterial,DATA  
c!MS$ END IF

      include '..\src\nodecl.inc'
      include '..\src\prop.inc'
      include '..\src\math.inc'
      include '..\src\anal_enu.inc'

      real*8    :: props(*)
      real*8    :: Coeff         ! global SF factor
      integer*4 :: IANA_IntInfo
      real*8    :: PRO_GetFromDat,PRO_GetParamNonl
      integer*4 :: iSFtype
      real*8    :: MultForCandSL,MultForTgPHI
      real*8    :: phi

      if ( IANA_IntInfo ( IANA_PROCESS_ACT ).eq.IANA_STABILITY ) then
        iSFtype = IPRO_GetStabilityTypeForMaterial ( props,Coeff,
     &                                      MultForCandSL,MultForTgPHI)
        select case ( iSFtype)
          case (IANA_SF_STRLEV)    ! stress-level based
            call FIBER_1D_ScaleCurves (props,Coeff)
          case (IANA_SF_TGPHIC)    ! tan(phi)-c based
            call FIBER_1D_ScaleCurves (props,Coeff)
          case (IANA_SF_COHES)     ! c-only based
            call FIBER_1D_ScaleCurves (props,Coeff)
        end select
      end if

      return
      end


c==========================================================================
      subroutine FIBER_1D_Model (order, props, model_buff, 
     &                           InfoIn, InfoOut,
     &                           domain)
c==========================================================================
      use FIBER_1D_model_module

!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
  !MS$ ATTRIBUTES DLLIMPORT :: /dat_common/   
!MS$ END IF

!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
  !MS$ ATTRIBUTES DLLIMPORT :: /JOB_common/
!MS$ END IF

!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
  !MS$ ATTRIBUTES DLLIMPORT :: /ELE_common/
!MS$ END IF

!!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
!  !MS$ ATTRIBUTES DLLIMPORT :: /ELG_common/
!!MS$ END IF
!
!!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
!  !MS$ ATTRIBUTES DLLIMPORT :: /ELE_current/
!!MS$ END IF

!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
  !MS$ ATTRIBUTES DLLIMPORT :: /CentralPoint_TRA/   
!MS$ END IF

!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
  !MS$ ATTRIBUTES DLLIMPORT :: /CurrentIntpoint_TRA/ 
!MS$ END IF

!MS$ IF DEFINED (__BUILD_DLL_ZCALC)
  !MS$ ATTRIBUTES DLLIMPORT :: /pointers_TRA/ 
!MS$ END IF


      include '..\src\nodecl.inc'
      include '..\src\prop.inc'
      include '..\src\dat.inc'
      include '..\src\job.inc'
      include '..\src\trans.inc'
      include '..\src\beam.inc'
c      include '..\src\elgeo.inc'
cNEW
      include '..\src\anal_enu.inc'

      integer*4, intent (in)    :: order
      real*8,    intent (in)    :: props      (*)
      integer*4, intent (inout) :: model_buff (*)
      integer*4, intent (in)    :: InfoIn     (*)
      integer*4, intent (inout) :: InfoOut    (*)
      integer*4, intent (inout) :: domain     (*)
     

      type (StateParameters_FIBER_1D) :: StateN
      type (StateParameters_FIBER_1D) :: StateNx1
      integer*4                  :: StateOffsInStorage
      integer*4                  :: StorageSize,j

      type (MatProperties_FIBER_1D)   :: Mat

      real*8                     :: E,C
      integer*4                  :: k,iorderTmp2,mech,loadingStatus
      integer*4                  :: n_T_pts,n_C_pts
      external FIBER_1D_CopyRightToLeft,FIBER_1D_CopyLeftToRight
      real*8                     :: PRO_GetFromDAT 

      integer*4 :: nstre
      real*8    :: strial(MAX_STRESS)
      logical*4 :: ANA_models_elastic_mode_ON,ANA_models_elastic_mode

      integer*4 :: layer,igaus,iele,imat,lprop_beam
      real*8    :: y,z,dA,dist

cNEW
      real*8    :: time,time_limits (2),t1,t2,A
      integer*4 :: exf,range

      integer*4 :: IELE_GET,IPRO_GET
      real*8    :: ANA_RealInfo

cNEW
      integer*4 :: IPRO_GetModelFromProps
      real*8    :: BEA_GetArea

      StorageSize = FIBER_1D_StateStorageSizeI4words () ! set pointers to state parameters
      StateOffsInStorage = StorageSize / 2
      nstre = 1 ! sets nr of of stress components for assumed job
                ! DON'T EDIT

      select case (order)

        ! --------------------
        case( L_GP_UPD_STATE )
        ! --------------------
          ! here you have to update state parameters like stresses, 
          ! plastic strains, hardening parameters, etc.............
          j = MODofs_DAT (MOD_ALL_DAT)+1 ! address in model storage buffer 
                                         ! DON'T EDIT

          call FIBER_1D_repack_state (model_buff(j),0,
     &                                StateN,FIBER_1D_CopyRightToLeft  )
          call FIBER_1D_repack_state  (model_buff(j),StateOffsInStorage,
     &                                StateNx1,FIBER_1D_CopyRightToLeft)

          StateN%Stress0 = StateNx1%Stress0
          ! here update all state variables:   old <= new 
          StateN = StateNx1

          call FIBER_1D_repack_state (model_buff(j),0,
     &                                StateN,FIBER_1D_CopyLeftToRight)
          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                                StateNx1,FIBER_1D_CopyLeftToRight)

        ! --------------------
        case( L_GP_NEW_STATE )
        ! --------------------
          ! additional data
          ! ---------------
          layer = ilayer_TRA - 1  ! IntLayer_ELG - 1  ! first is the whole cross section for shear
          igaus = igaus_TRA      ! IntPoint_ELG
          iele  = iele_TRA       ! ICurrNr_ELE
          ! beam element material
          imat = IELE_Get( IELE_IMAT,iele,domain)
cNEW
          time = ANA_RealInfo (IANA_TIME_ACT)
          exf  = IPRO_Get( IPRO_EXF,imat,domain)
          if ( exf.gt.0 ) then
            range = 1
            call EXF_GetRec (domain,exf,range,time_limits)
            t1 = time_limits (1)
            t1 = dmax1 (0.0d0,t1)
            t2 = time_limits (2)
          end if

          lprop_beam = IPRO_Get( IPRO_DATPTR,imat,domain )
          ! layer position in the cross section
          y = PRO_GetFromDat(domain(lprop_beam),IPRO_GR_GEOM,
     &               12+(layer-1)*IBEA_GEOM_INFOSIZE+IBEA_GEOM_Y)
          z = PRO_GetFromDat(domain(lprop_beam),IPRO_GR_GEOM,
     &               12+(layer-1)*IBEA_GEOM_INFOSIZE+IBEA_GEOM_Z)
          ! layer area 
          dA= PRO_GetFromDat(domain(lprop_beam),IPRO_GR_GEOM,
     &               12+(layer-1)*IBEA_GEOM_INFOSIZE+IBEA_GEOM_AREAYZ)
          ! distance between beams in 2D
          dist = PRO_GetFromDat (domain(lprop_beam),IPRO_GR_GEOM,
     &                           IBEA_GEOM_DIST)
          dA = dA / dist

           ! compute new state for given effective strain increment dE_TRA
           !         (dE_TRA = dE - dEo - dEcr)
           !         where dE   - total strain increment
           !         where dE0  - initial strain increment
           !         where dEcr - creep strain increment
           !                             initial stress increment   dSig0_TRA
           !                             initial elastic moduli     Dev_TRA

          j = MODofs_DAT (MOD_ALL_DAT)+1 ! address in model storage buffer 
                                         ! DON'T EDIT

          call FIBER_1D_repack_state (model_buff(j),0,
     &                    StateN,FIBER_1D_CopyRightToLeft  )
          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                    StateNx1,FIBER_1D_CopyRightToLeft)

          if ( ANA_models_elastic_mode_ON ()) then
            ! needed for eigenvalue analysis
c            call FIBER_1D_unpack_props (props,Mat)
            call FIBER_1D_modify_soft (Mat,he_TRA)
            !E = FIBER_1D_E_secant ( StateN,Mat )
            E = FIBER_1D_E_init ( Mat )
            strial = StateN % Stress + E * dE_TRA(1)
            Dev_TRA (1) = E
            if ( ANA_models_elastic_mode (strial,nstre,
     &                                StateNx1 % Stress,
     &                                Sact_tra,
     &                                Dev_TRA,Dep_TRA)) then
              return
            end if
          end if

          call FIBER_1D_modify_soft (Mat,he_TRA)

          call FIBER_1D_NewState(dE_TRA(1),dSig0_TRA(1),StateN,StateNx1,
     &                           Dev_TRA(1),Dep_TRA(1),Mat)

          Sact_TRA (1) = StateNx1 % Stress 

          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                                StateNx1,FIBER_1D_CopyLeftToRight)

        ! --------------------
        case( L_GP_INI_STATE )
        ! --------------------
          ! initialization of state parameters
          ! nulify all your arrays
          j = MODofs_DAT (MOD_ALL_DAT)+1 ! address in model storage buffer 
                                         ! DON'T EDIT
          call FIBER_1D_repack_state (model_buff(j),0,
     &                    StateN,FIBER_1D_CopyRightToLeft  )
          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                    StateNx1,FIBER_1D_CopyRightToLeft  )

          call FIBER_1D_unpack_props (props,Mat)       
          
          call FIBER_1D_modify_soft  (Mat,he_TRA)         

          call FIBER_1D_InitStateParameters (StateN,StateNx1,Mat)

          call FIBER_1D_repack_state (model_buff(j),0,
     &                               StateN,FIBER_1D_CopyLeftToRight )
          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                               StateNx1,FIBER_1D_CopyLeftToRight)

        ! --------------------
        case( L_GP_SIZE      )
        ! --------------------
          ! return through output buffer the ammount of INT*4 words needed
          ! to save all your state parameters (the float values keep always
          ! as real*8 words which is equivalent to 2 words of INT*4 
          InfoOut (MOD_ALL_DAT) = StorageSize ! DON'T EDIT

        ! --------------------
        case( L_GP_IFSYM     )
        ! --------------------
          ! return the information whether your tangent stiffness matrix 
          ! is symmetric (if yes put 1 and if not put 0)
          InfoOut(1) = 1        ! symetric

        ! --------------------
        case( L_GP_SETDELA   )
        ! --------------------
          ! return the actual elastic stiffness  constitutive matrix D
          ! through the output buffer
          ! if the elastic stiffness is a function of stress 
          ! then perform an action:
          !
          ! offs = MODofs_DAT (MOD_ALL_DAT) ! DON'T EDIT
          ! compute stiffness matrix based on actual stress
          j = MODofs_DAT (MOD_ALL_DAT)+1 ! address in model storage buffer 
                                         ! DON'T EDIT
          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                    StateNx1,FIBER_1D_CopyRightToLeft  )
          call FIBER_1D_unpack_props (props,Mat)
          !E = FIBER_1D_E_secant ( StateNx1,Mat )
          E = FIBER_1D_E_init ( Mat )
          call UTL_move ( InfoOut,E,1 )

        ! --------------------
        case( L_GP_SETCELA   )
        ! --------------------
          ! return the actual elastic compliance constitutive matrix C
          ! through the output buffer

          ! if the elastic stiffness is a function of stress 
          ! then perform an action:
          !
          ! offs = MODofs_DAT (MOD_ALL_DAT) ! DON'T EDIT
          ! compute compliance matrix based on actual stress

          j = MODofs_DAT (MOD_ALL_DAT)+1 ! address in model storage buffer 
                                         ! DON'T EDIT
          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                    StateNx1,FIBER_1D_CopyRightToLeft  )
          call FIBER_1D_unpack_props (props,Mat)
          !E = FIBER_1D_E_secant ( StateNx1,Mat )
          E = FIBER_1D_E_init ( Mat )
          C = 1.0d0/E
          call UTL_move ( InfoOut,C,1 )

        ! --------------------
        case( L_GP_PRINTPARAM )
        ! --------------------
          call FIBER_1D_PrintProps(Props, InfoIn)

c        ! --------------------
c        case ( L_GP_SETDATASIZE ) 
c        ! --------------------
c          if (InfoIn(1).eq.IPRO_GR_NONL) then
c            n_T_pts = PRO_GetFromDAT ( props,IPRO_GR_NONL,
c     &                           FIBER_1D_NONL_N_TENSILE_CURVE_PTS)
c            n_C_pts = PRO_GetFromDAT ( props,IPRO_GR_MAIN,
c     &                           FIBER_1D_NONL_N_COMPRES_CURVE_PTS)
c            InfoOut(1)= (n_T_pts+n_C_pts)*2
c          else
c            call sys_stop ("FIBER_1D_Model - internal error")
c          end if

        ! --------------------
        case (L_GP_GET_INFO)
        ! --------------------
          ! return through transfer common block (see trans.inc file)
          ! all requested results/data
          call FIBER_1D_unpack_props (props,Mat)
          j = MODofs_DAT (MOD_ALL_DAT)+1 ! address in model storage buffer 
                                         ! DON'T EDIT
          call FIBER_1D_repack_state (model_buff(j),StateOffsInStorage,
     &                    StateNx1,FIBER_1D_CopyRightToLeft)
          k = 1
          do while (k.le.ITRA_NR_OF_ITEM)
            iorderTmp2 = ishft(1,k-1)
            if( iand(InfoIn(1), iorderTmp2) .ne. 0 ) then
              select case (iorderTmp2)
                case ( ITRA_GP_SACT )
                  Sact_TRA (1) = StateNx1 % Stress 
                case ( ITRA_GP_POISS )
                case ( ITRA_GP_IPLA )
                  Iplas_TRA = 0
                  StatusChar_TRA ='  E  '
                case ( ITRA_GP_SLEV )
                  SLev_TRA = 0.0d0
                case ( ITRA_GP_YOUNG)
                case default
                  continue 
              end select
            end if
            k=k+1
          end do

        ! --------------------
        case ( L_GP_MODIFPARAM )
        ! --------------------

        ! --------------------
        case ( L_GP_SET_INCOMP)
        ! --------------------

        ! --------------------
        case ( L_GP_FORM_DESCR ) 
        ! --------------------
          InfoOut (IPRO_FORMULATION) = IPRO_FORM_ELASTO_PLAST

        ! --------------------
        case default
        ! --------------------

      end select


      return
          end

c========================================================================
      integer*4 function FIBER_1D_softening_flag (props)
c========================================================================
      use FIBER_1D_model_module
      include 'nodecl.inc'
      real*8 :: props (*)
      type (MatProperties_FIBER_1D) :: Mat
      integer*4 :: i

      call FIBER_1D_unpack_props (props,Mat)

      do i=Mat % n_T_pts,2,-1
        if (         abs(Mat % T_curve (2,i)).lt.
     &      0.99d0 * abs(Mat % T_curve (2,i-1))) then
          FIBER_1D_softening_flag = 1
          return
        end if
      end do

      do i=Mat % n_C_pts,2,-1
        if (         abs(Mat % C_curve (2,i)).lt.
     &      0.99d0 * abs(Mat % C_curve (2,i-1))) then
          FIBER_1D_softening_flag = 1
          return
        end if
      end do

      FIBER_1D_softening_flag = 0

      return
      end function FIBER_1D_softening_flag



c========================================================================
      integer*4 function FIBER_1D_softening_flag_ex (Mat)
c========================================================================
      use FIBER_1D_model_module
      include 'nodecl.inc'
      type (MatProperties_FIBER_1D) :: Mat
      integer*4 :: i
      integer*4 :: ret

      ret = 0

      do i=Mat % n_T_pts,2,-1
        if (         abs(Mat % T_curve (2,i)).lt.
     &      0.99d0 * abs(Mat % T_curve (2,i-1))) then
          ret = 1
          go to 111
        end if
      end do

111   continue

      do i=Mat % n_C_pts,2,-1
        if (         abs(Mat % C_curve (2,i)).lt.
     &      0.99d0 * abs(Mat % C_curve (2,i-1))) then
          ret = ior (ret,2)
          go to 222
        end if
      end do

222   continue

      FIBER_1D_softening_flag_ex = ret

      return
      end function FIBER_1D_softening_flag_ex

c========================================================================
      subroutine FIBER_1D_modify_soft (Mat,he)
c========================================================================
      use FIBER_1D_model_module
      include 'nodecl.inc'
      type (MatProperties_FIBER_1D) :: Mat
      real*8                        :: he
      integer*4                     :: soft_flag
      integer*4                     :: FIBER_1D_softening_flag_ex
      real*8                        :: eps_soft

      soft_flag = FIBER_1D_softening_flag_ex (Mat)
      if ( Mat % Lc.le.1.0d-20 ) then
        return
      end if

      if ( iand(soft_flag,1).ne.0 ) then
        call FIBER_1D_modify_soft_curve (Mat % n_T_pts, Mat % T_curve,
     &                                   Mat % Lc, he, Mat % ET_o)
      end if

      if ( iand(soft_flag,2).ne.0 ) then
        call FIBER_1D_modify_soft_curve (Mat % n_C_pts, Mat % C_curve,
     &                                   Mat % Lc, he, Mat % EC_o)
      end if      

      end subroutine FIBER_1D_modify_soft


c========================================================================
      subroutine FIBER_1D_modify_soft_curve ( size,curve,Lc,he,Eo)
c========================================================================
      include 'nodecl.inc'
      integer*4 :: size
      real*8    :: curve (2,*)
      real*8    :: Lc,he,Eo
      real*8    :: eps1,eps2,sig1,sig2,sig_max,epso,deps
      integer*4 :: i,beg_point,end_point

      if ( size.le.1 ) then
        return
      end if

      ! find maximum stress value
      sig_max = 0.0d0
      do i=1,size
        sig_max = dmax1 ( sig_max,dabs(curve (2,i)))
      end do

      ! find starting point of softening branch 

c     eps1 = curve (1,1)
      sig1 = dabs(curve (2,1))

      beg_point = 0
      !end_point = 0

      do i=2,size
c       eps2 = curve (1,i)
        sig2 = dabs(curve (2,i))
        if ( beg_point.eq.0 ) then
          if ( sig2-sig1.lt.-1.0d-3*sig_max ) then
            beg_point = i-1
            go to 111
          end if
        end if
        sig1 = sig2
c       eps1 = eps2
      end do  

111   continue

      if ( beg_point.ne.0 ) then
        !H = 0.0d0
        eps1 = curve (1,beg_point)
        sig1 = curve (2,beg_point)
        epso = curve (1,beg_point)

        do i=beg_point+1,size
          eps2 = curve (1,i)
          sig2 = curve (2,i)
          deps = eps2 - epso
          deps = deps * Lc / he
          curve (1,i) = epso + deps
          !dsig = sig2-sig1
          !deps = eps2-eps1
          !H = dmin1 (H,dsig/deps)
        end do         
      end if

      end subroutine FIBER_1D_modify_soft_curve


c ==============================================================================
      subroutine FIBER_1D_scale_tensile_branch (Mat,eps_compr_max)
c ==============================================================================
      use FIBER_1D_model_module
      include 'nodecl.inc'
      type (MatProperties_FIBER_1D) :: Mat
      real*8 :: eps_compr_max
      real*8 :: Stress,Et,sig_max,coeff
      integer*4 :: i

      if ( Mat % Coupled_compr_tens_softening.gt.0 ) then
        if ( Mat % Compr_soft_is_possible.gt.0 ) then
          if ( dabs(eps_compr_max).gt.dabs(Mat % Compr_soft_eps) ) then
            ! find maximum sigma on compression branch
            sig_max = 0.0d0
            do i=1,Mat % n_C_pts
              sig_max = dmax1 (sig_max,dabs(Mat % C_curve (2,i)))
            end do
            sig_max = dmax1 (sig_max,1.0d-20)

            call FIBER_1D_InterpolateCurve (eps_compr_max,Mat % C_curve,
     &                                      Mat % n_C_pts,Stress,Et) 
            coeff = Stress / sig_max

            do i=1,Mat % n_T_pts
              Mat % T_curve (2,i) = Mat % T_curve (2,i) * coeff
            end do

          end if
        end if
      end if

      end subroutine FIBER_1D_scale_tensile_branch 


          