#ifndef __CROSS_SECTION_H
#define __CROSS_SECTION_H
#pragma once

#include "zmate_hdr.h"

#include <vector>
#include "../ZUTL/MeshGeom.hpp"

class CReinforcedSectionPlot;
class CableSectionPlot;
class Beam_Reinforcement_Set;
class Beam_Reinforcement_Sets;
class BeamCableDef;
class BeamCableBase;
class T3DPoint;
class DimLine;
class CableGeometry;


class _ZMATE_CLASS CrossSection
  {
  public:
  enum {CROSS_SECTION_UNDEFINED ,
        CROSS_SECTION_RECT,
        CROSS_SECTION_RECT_TUBE ,
        CROSS_SECTION_CIRCLE    ,
        CROSS_SECTION_CIRC_TUBE ,
        CROSS_SECTION_RECT_BOX  ,
        CROSS_SECTION_I_SHAPE_S ,
        CROSS_SECTION_I_SHAPE_NS,
        CROSS_SECTION_T_SHAPE   ,
        CROSS_SECTION_RECT_AXI,
        CROSS_SECTION_MAX      };

    CrossSection  ();
    ~CrossSection ();

    virtual CrossSection *GetCopy() { return new CrossSection(); }

    virtual int  type (){ return CROSS_SECTION_UNDEFINED;}
    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set) {}

    virtual int IsInside (Beam_Reinforcement_Set* set) { return 0;}
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v) { return 0;}
    virtual int GetYZ2D(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v) { return 0;}
    virtual void GetY(CableGeometry* set,T3DPoint &pt) {}
    virtual void GetZ(CableGeometry* set,T3DPoint &pt);
    virtual void GetY(CableGeometry* set,double coord,T3DPoint &pt) {}
    virtual void GetZ(CableGeometry* set,double coord,T3DPoint &pt) {}

    virtual int IsCircular() { return 0; }

    virtual CString GetSectionName() { return CString("Base"); }

    virtual double GetSectionWidth(double ypos) { return 0.0;}
    virtual double GetSectionWidth(Beam_Reinforcement_Set* set) { return 0.0;}

    virtual double inertia_x  () { return 0.0;}
    virtual double inertia_y  () { return 0.0;}
    virtual double inertia_z  () { return 0.0;}
    virtual double inertia_yz () { return (inertia_y()+inertia_z()); }


    virtual double area_x     () { return 0.0;}
    virtual double area_y     () { return 0.0;}
    virtual double area_z     () { return 0.0;}

    virtual double height_y   () { return 0.0;}
    virtual double height_z   () { return 0.0;}

    virtual double height_y_top () { return 0.0;}
    virtual double height_y_bottom () { return 0.0;}

    virtual void Discretize(MeshGeom &mesh) {}

    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines) {}

    virtual void GetCornerPointsYZ(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2);
  };


class _ZMATE_CLASS CrossSection_RECT : public CrossSection
  {
  public:
    double b,h;
  public:
    CrossSection_RECT  (double b, double h);
    ~CrossSection_RECT ();

    virtual CrossSection *GetCopy() { return new CrossSection_RECT(*this); }

    int type (){ return CrossSection::CROSS_SECTION_RECT;}
    virtual CString GetSectionName() { return CString("Plane rectangle"); }
    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);

    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual int GetYZ2D(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetY(CableGeometry* set,T3DPoint &pt);
    virtual void GetY(CableGeometry* set,double coord,T3DPoint &pt);
    virtual void GetZ(CableGeometry* set,double coord,T3DPoint &pt);

    virtual double GetSectionWidth(double ypos) { return b; }
    virtual double GetSectionWidth(Beam_Reinforcement_Set* set) { return b;}

    virtual double inertia_x  (); 
    virtual double inertia_y  (); 
    virtual double inertia_z  (); 

    virtual double area_x     (); 
    virtual double area_y     (); 
    virtual double area_z     (); 

    virtual double height_y   () { return b;}
    virtual double height_z   () { return h;} 

    virtual double height_y_top () { return b;} 
    virtual double height_y_bottom () { return b;}

    virtual void Discretize(MeshGeom &mesh);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
};


class _ZMATE_CLASS CrossSection_RECT_XY : public CrossSection_RECT
{
public:
 CrossSection_RECT_XY  (double b, double h) : CrossSection_RECT(b,h) {}
    
 virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                             std::vector<DimLine> &dimLines);
};

class _ZMATE_CLASS CrossSection_RECT_XZ : public CrossSection_RECT
{
public:
 CrossSection_RECT_XZ  (double b, double h) : CrossSection_RECT(b,h) {}

 virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
};



class _ZMATE_CLASS CrossSection_RECT_Axi : public CrossSection_RECT
  {
  public:
    CrossSection_RECT_Axi  (double _h);
    ~CrossSection_RECT_Axi () {}

    virtual CrossSection *GetCopy() { return new CrossSection_RECT(*this); }

    int type (){ return CrossSection::CROSS_SECTION_RECT_AXI;}
    virtual CString GetSectionName() { return CString("Plane rectangle"); }
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
  };

class _ZMATE_CLASS CrossSection_RECT_TUBE : public CrossSection
  {
  public:
    double b,h,th;
  public:
    CrossSection_RECT_TUBE  (double b, double h,double th);
    ~CrossSection_RECT_TUBE ();
    virtual CrossSection *GetCopy() { return new CrossSection_RECT_TUBE(*this); }

    int type (){ return CROSS_SECTION_RECT_TUBE;}
    virtual CString GetSectionName() { return CString("Rectangular tube"); }

    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);

    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetY(CableGeometry* set,T3DPoint &pt);
    virtual void GetY(CableGeometry* set,double coord,T3DPoint &pt);
    virtual void GetZ(CableGeometry* set,double coord,T3DPoint &pt);

    virtual double GetSectionWidth(Beam_Reinforcement_Set* set) { return b;}

    virtual double inertia_x  (); 
    virtual double inertia_y  (); 
    virtual double inertia_z  (); 

    virtual double area_x     (); 
    virtual double area_y     (); 
    virtual double area_z     (); 

    virtual double height_y   () { return b;}
    virtual double height_z   () { return h;} 

    virtual double height_y_top () { return b;} 
    virtual double height_y_bottom () { return b;}

    virtual void Discretize(MeshGeom &mesh);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
};

class _ZMATE_CLASS CrossSection_CIRCLE : public CrossSection
  {
  public:
    double d;
  public:
    CrossSection_CIRCLE  (double d);
    ~CrossSection_CIRCLE ();
    virtual CrossSection *GetCopy() { return new CrossSection_CIRCLE(*this); }

    int type (){ return CROSS_SECTION_CIRCLE;}
    virtual CString GetSectionName() { return CString("Plane circular"); }

    virtual int IsCircular() { return 1; }

    virtual double GetSectionWidth(Beam_Reinforcement_Set* set) { return d;}
    virtual double height_y   () { return d;}
    virtual double height_z   () { return d;} 

    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);
    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
  };

class _ZMATE_CLASS CrossSection_CIRC_TUBE : public CrossSection
  {
  public:
    double d,th;
  public:
    CrossSection_CIRC_TUBE  (double d,double th);
    ~CrossSection_CIRC_TUBE ();
    virtual CrossSection *GetCopy() { return new CrossSection_CIRC_TUBE(*this); }
    int type (){ return CROSS_SECTION_CIRC_TUBE;}
    virtual CString GetSectionName() { return CString("Circular tube"); }
    virtual int IsCircular() { return 1; }

    virtual double GetSectionWidth(Beam_Reinforcement_Set* set) { return d;}
    virtual double height_y   () { return d;}
    virtual double height_z   () { return d;} 

    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);
    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
  };


class _ZMATE_CLASS CrossSection_RECT_BOX : public CrossSection
  {
  public:
    double b,h,tb,th;
  public:
    CrossSection_RECT_BOX  (double b,double h,double tb,double th);
    ~CrossSection_RECT_BOX ();
    virtual CrossSection *GetCopy() { return new CrossSection_RECT_BOX(*this); }
    int type (){ return CROSS_SECTION_RECT_BOX;}
    virtual CString GetSectionName() { return CString("Box"); }

    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);

    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetY(CableGeometry* set,T3DPoint &pt);
    virtual void GetY(CableGeometry* set,double coord,T3DPoint &pt);
    virtual void GetZ(CableGeometry* set,double coord,T3DPoint &pt);

    virtual double GetSectionWidth(Beam_Reinforcement_Set* set) { return b;}

    virtual double inertia_x  (); 
    virtual double inertia_y  (); 
    virtual double inertia_z  (); 

    virtual double area_x     (); 
    virtual double area_y     (); 
    virtual double area_z     (); 

    virtual double height_y   () { return b;}
    virtual double height_z   () { return h;} 

    virtual double height_y_top () { return b;} 
    virtual double height_y_bottom () { return b;}

    virtual void Discretize(MeshGeom &mesh);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
  };

class _ZMATE_CLASS CrossSection_I_SHAPE_S : public CrossSection
  {
  public:
    double b,h,tw,tf;
  public:
    CrossSection_I_SHAPE_S  (double d,double h,double tb,double th);
    ~CrossSection_I_SHAPE_S ();
    virtual CrossSection *GetCopy() { return new CrossSection_I_SHAPE_S(*this); }
    int type (){ return CROSS_SECTION_I_SHAPE_S;}
    virtual CString GetSectionName() { return CString("I-bisym"); }

    virtual double GetSectionWidth(Beam_Reinforcement_Set* set);

    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);

    virtual double height_y   () { return b;}
    virtual double height_z   () { return h;} 

    virtual double height_y_top () { return b;} 
    virtual double height_y_bottom () { return b;}

    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetY(CableGeometry* set,T3DPoint &pt);
    virtual void GetY(CableGeometry* set,double coord,T3DPoint &pt);
    virtual void GetZ(CableGeometry* set,double coord,T3DPoint &pt);

    virtual void Discretize(MeshGeom &mesh);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
  };

class _ZMATE_CLASS CrossSection_I_SHAPE_NS : public CrossSection
  {
  public:
    double b1,b2,tf1,tf2,h,tw;
  public:
    CrossSection_I_SHAPE_NS  (double b1,double b2,double tf1,double tf2,double h,double tw);
    ~CrossSection_I_SHAPE_NS ();
    virtual CrossSection *GetCopy() { return new CrossSection_I_SHAPE_NS(*this); }
    int type (){ return CROSS_SECTION_I_SHAPE_NS;}
    virtual CString GetSectionName() { return CString("I-asym"); }
    double Get_Zo ();

    virtual double GetSectionWidth(Beam_Reinforcement_Set* set);

    virtual double height_y   () { return b1;}
    virtual double height_z   () { return h;} 

    virtual double height_y_top () { return b1;} 
    virtual double height_y_bottom () { return b2;}

    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);

    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetY(CableGeometry* set,T3DPoint &pt);
    virtual void GetY(CableGeometry* set,double coord,T3DPoint &pt);
    virtual void GetZ(CableGeometry* set,double coord,T3DPoint &pt);

    virtual void Discretize(MeshGeom &mesh);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
  };

class _ZMATE_CLASS CrossSection_T_SHAPE : public CrossSection
  {
  public:
    double b,tw,h,tf;
  public:
    CrossSection_T_SHAPE  (double b,double tw,double h,double tf);
    ~CrossSection_T_SHAPE ();
    virtual CrossSection *GetCopy() { return new CrossSection_T_SHAPE(*this); }
    int type (){ return CROSS_SECTION_T_SHAPE;}
    virtual CString GetSectionName() { return CString("T-section"); }
    double Get_Zo ();

    virtual double GetSectionWidth(Beam_Reinforcement_Set* set);

    virtual void Draw (CReinforcedSectionPlot *sectPlot,Beam_Reinforcement_Sets *set);

    virtual double height_y   () { return b;}
    virtual double height_z   () { return h;} 

    virtual double height_y_top () { return b;} 
    virtual double height_y_bottom () { return tw;}

    virtual int IsInside (Beam_Reinforcement_Set* set);
    virtual int GetYZ(Beam_Reinforcement_Set* set,std::vector<T3DPoint> &v);
    virtual void GetY(CableGeometry* set,T3DPoint &pt);
    virtual void GetY(CableGeometry* set,double coord,T3DPoint &pt);
    virtual void GetZ(CableGeometry* set,double coord,T3DPoint &pt);

    virtual void Discretize(MeshGeom &mesh);
    virtual void GetPlotObjects(std::vector<T3DPoint> &cornerPts,std::vector<T3DPoint> &cornerPts2,
                                std::vector<DimLine> &dimLines);
  };


#endif // !defined(__CROSS_SECTION_H)