#ifndef T3DPoint_h
#define T3DPoint_h

#include "zutl_hdr.h"
#include "stdio.h"
#include "..\\H\\constant.h"

#include "memory.h"
#include <math.h>

class T3DVector;
class T3DLine ;

class _ZUTL_CLASS T3DPoint 
{
protected:
	double X[3];

public:

	friend class T3DVector;

/*@Public_Methods*/
/*@Section Constructor and Destructor*/
/**/
	T3DPoint() ;
	T3DPoint(const T3DPoint &) ;
	T3DPoint(T3DPoint *) ;
	T3DPoint( double, double, double) ;
	T3DPoint(const double *)  ;
	T3DPoint(Vector *) ;
	T3DPoint(T3DVector) ;
	~T3DPoint();
	void initialise() ;
   	
/*@Section Operators*/
/**/
	T3DPoint operator+(const T3DVector &) const;
	T3DPoint operator-(const T3DVector &) const;
	T3DPoint operator+(const T3DPoint &) const;
	T3DPoint operator-(const T3DPoint &) const;
	T3DPoint operator/(double) const;
	T3DPoint operator*(double) const ;

 //	T3DPoint * operator+(const T3DVector &) ;
//	T3DPoint * operator-(const T3DVector &)  ;

	T3DPoint& operator +=(const T3DPoint &) ;
	T3DPoint& operator -=(const T3DPoint &) ;
	T3DPoint& operator /=(double) ;
	int operator==(const T3DPoint) const ;
	double operator[] (int) const ;
	double& operator[](int n) { return X[n]; }


/*@Section Accessing variables */
/**/
	void SetPoint(const double *) ;
	void SetPoint(double, double, double) ;
	void SetPoint(const T3DVector) ;
	double getX() ;
	double getY() ;
	double getZ() ;	   
	double* getPoint() {return X;}	   
	void set(Vector) ;
	BOOL isAt(double, double, double) ;
	BOOL isAt(T3DPoint *) ;
	BOOL isInSphereOf(const T3DPoint &, double) ;
	BOOL isInBoxOf(const T3DPoint &,const T3DPoint &) ;
	BOOL isBetweenPoints(const T3DPoint &p1,const T3DPoint &p2);
 void SetBox(T3DPoint &pmin, T3DPoint &pmax);
 double getNthCoordinate(int) ;
	void SetNthCoordinate(int, double) ; 
	Vector asVector() ;
	T3DVector asT3DVector() ;

/*@Section Utilities */
/**/
	void multiply(double) ;
	void multiplyDirectionBy(int, double) ;

	void add(const T3DVector &) ;
	void sub(const T3DVector &) ;

	double computeDistanceToPoint(T3DPoint *) ;
	double computeDistanceToPoint(const T3DPoint &) ;
	double computeDistanceToPoint(double, double, double) ;

	_ZUTL_FUN friend double Distance(const T3DPoint &,const T3DPoint &);
	_ZUTL_FUN friend void computeBoxPoints(T3DPoint * pmin, T3DPoint * pmax);
	_ZUTL_FUN friend int isPointInBox(T3DPoint *pmin, T3DPoint * pmax,T3DPoint p);

	T3DPoint computeRotationVector(T3DLine line, double angle) ;

};
/*****************************************************************************/
/*               I N L I N E   M E T H O D S   D E F I N I T I O N           */
/*****************************************************************************/
/*@Method
	Create a new instance of `T3DPoint` object at (0, 0, 0).
*/
inline T3DPoint::T3DPoint() 
{	initialise() ;
}

/*@Method
	Create a new instance of `T3DPoint` object from `point`.
*/
inline T3DPoint::T3DPoint(const T3DPoint & point)
{	initialise() ;
	X[0] = point.X[0] ;
	X[1] = point.X[1] ;
	X[2] = point.X[2] ;
}

/*@Method
	Create a new instance of `T3DPoint` object from `point`.
*/
inline T3DPoint::T3DPoint(T3DPoint * point)
{	initialise() ;
	X[0] = point->X[0] ;
	X[1] = point->X[1] ;
	X[2] = point->X[2] ;
}				  

/*@Method
	Create a new instance of `T3DPoint` object from `point`.
*/
inline T3DPoint::T3DPoint(double x, double y, double z)
{	initialise() ;
	X[0] = x;
	X[1] = y ;
	X[2] = z ;
}
/*@Method
	Create a new instance of `T3DPoint` object from `vector`.
*/
inline T3DPoint::T3DPoint(const double*p)  
{ 	SetPoint(p); 
}


/*@Method
	Initialise coordinates to zero.
*/
inline void T3DPoint::initialise()
{	X[0] = X[1] = X[2] = 0.0f ;
}

/*@Method
	Delete this object.
*/
inline T3DPoint::~T3DPoint()
{	;
}

inline void T3DPoint::SetPoint(const double *p) 
{ 	X[0] = p[0]; 
	X[1] = p[1]; 
	X[2] = p[2]; 
}

inline void T3DPoint::SetPoint(double x, double y, double z ) 
{ 	X[0] = x; 
	X[1] = y; 
	X[2] = z; 
}

inline void T3DPoint::SetNthCoordinate(int i, double value) 
{ 	X[i] = value ; 
}

/*@Method
	Return `x` coordinate of this object.
*/
inline double T3DPoint::getX()
{	return X[0] ;
}

/*@Method
	Return `y` coordinate of this object.
*/
inline double T3DPoint::getY()
{	return X[1] ;
}

/*@Method
	Return `z` coordinate of this object.
*/
inline double T3DPoint::getZ()
{	return X[2] ;
}

inline T3DPoint T3DPoint::operator*(double val) const 
{ 	return T3DPoint( X[0]*val,X[1]*val,X[2]*val); 
}

inline T3DPoint & T3DPoint::operator +=(const T3DPoint & p) 
{ 	(*this)=(*this)+p; 
	return *this; 
}

inline T3DPoint & T3DPoint::operator -=(const T3DPoint & p) 
{ 	(*this)=(*this)-p; 
	return *this; 
}

inline T3DPoint& T3DPoint::operator /=(double val) 
{ 	X[0] /= val; 
	X[1] /= val; 
	X[2] /= val; 
	return *this; 
}

inline int T3DPoint::operator==(const T3DPoint p) const
{
 double epsilon = 1.0E-8 ;
	return ((fabs(X[0]-p.X[0]) < epsilon) && (fabs(X[1]-p.X[1]) < epsilon) && (fabs(X[2]-p.X[2]) < epsilon)) ;
}

inline double T3DPoint::operator[] (int n) const
{ 	return X[n]; 
}

/*@Method
	Compute and return the distance between this object and `point`.
*/
inline double T3DPoint::computeDistanceToPoint(T3DPoint * point)
{	return computeDistanceToPoint(point->X[0], point->X[1], point->X[2]) ;
}

inline double T3DPoint::computeDistanceToPoint(const T3DPoint& point)
{	return computeDistanceToPoint(point.X[0], point.X[1], point.X[2]) ;
}

/*@Method
	Set the coordinates of this object to `x1`, `y1` and `z1`.
*/
inline void T3DPoint::set(Vector v)
{	X[0] = (double)v.x ;
	X[1] = (double)v.y ;
	X[2] = (double)v.z ;
}
/*@Method
	Return nth coordinates.
*/
inline double T3DPoint::getNthCoordinate(int n)
{	if (n==1) return X[0];
	else if (n==2) return X[1] ;
	else if (n==3) return X[2] ;
	return 0.00	;
} 
//////////////////////////////////////////////////////////////////////////
inline BOOL T3DPoint::isAt(T3DPoint * point)
//////////////////////////////////////////////////////////////////////////
{	return isAt(point->X[0], point->X[1], point->X[2]) ;
}


#endif /* T3DPoint_h */
