// Handling of units 

#ifndef UNIT_H
#define UNIT_H

#include "zutl_hdr.h"
#include <vector>

#include "../H/bag.h"
#include "../H/z_fstream.h"
#include "../H/UNIT_fieldBase.h"


using std::vector;

class UNIT_manager;
class UNIT_Simple;
class LinkedList;

enum UNIT_baseUnits
{
	UNIT_BASE_UNDEF = -1,
	UNIT_BASE_FORCE,//0
	UNIT_BASE_LENGTH,//1
	UNIT_BASE_ANGLE,//2
	UNIT_BASE_TIME, //3
	UNIT_BASE_TEMP, //4
	UNIT_BASE_MASS, //6
	UNIT_BASE_ENERGY, //7
	UNIT_BASE_POWER,  //8
	UNIT_BASE_NROF_ALL
};

#define UNIT_BASE_NROF UNIT_BASE_MASS


enum UNIT_ComplexUnits
{
	UNIT_DIMLESS = 0,                   // 0
	UNIT_FORCE,                     // 1 
	UNIT_LENGTH,                     // 2 
	UNIT_ANGLE,                     // 3 
	UNIT_TIME,                     // 4 
	UNIT_TEMP,                     // 5 
	UNIT_INVLENGTH,                   // 6 
	UNIT_AREA,                     // 7 
	UNIT_VOLUME,                     // 8 
	UNIT_INERTIA,                     // 9 
	UNIT_FORC_PER_LEN,               // 10 
	UNIT_STRESS,                     // 11 
	UNIT_FORC_PER_VOL,               // 12
	UNIT_MOMENT,                     // 13 
	UNIT_VELOC,                     // 14 
	UNIT_INVTEMP,                 // 15 
	UNIT_HEAT_CONDUCTIVITY_GLOB,  // 16 
	UNIT_HEATSRC,                 // 17 
	UNIT_HEATCAP_GLOB,             // 18 
	UNIT_HEATFLUX,                 // 19 
	UNIT_HEAT_CONV_COEF_GLOB,       // 20 
	UNIT_HUMDIFF,                 // 21 
	UNIT_INVTIME,                 // 22 
	UNIT_HUMIDFLUX,                 // 23 
	UNIT_VOLPERTIME,                 // 24 
	UNIT_HEATFLUXAREA,                 // 25 
	UNIT_HEATFLUX_PER_LEN,             // 26 
	UNIT_HUMIDFLUX_PER_LEN,            // 27 
	UNIT_AREAPERTIME,                  // 28 
	UNIT_FORCEPERTIME,                 // 29 
	UNIT_HUMIDFLUX_LEN,                // 30 
	UNIT_INVSTRESS,                   // 31 
	UNIT_TEMP_PER_LEN,                 // 32 
	UNIT_ACCELERATION,                 // 33 
	UNIT_MASS,                         // 34 
	UNIT_MASS_PER_LEN,                 // 35 
	UNIT_INERTIA_ROT,                  // 36 
	UNIT_ANGLE_VEL,                    // 37
	UNIT_ANGLE_ACC,                    // 38
	UNIT_MASS_PER_AREA,                // 39 
	UNIT_MASS_PER_VOL,                 // 40 
	UNIT_STEFAN_BOLTZMANN,             // 41
	UNIT_SEEPAGE,                      // 42
	UNIT_INV_ANGLE,                    // 43
	UNIT_ENERGY_GLOB,                  // 44
	UNIT_POWER_GLOB,                   // 45

	UNIT_MASS2,                         // 46
	UNIT_MASS_PER_LEN2,                 // 47
	UNIT_INERTIA_ROT2,                  // 48
	UNIT_MASS_PER_AREA2,                // 49
	UNIT_MASS_PER_VOL2,                 // 50

	UNIT_ENERGY,                        // 51
	UNIT_POWER,                         // 52
	UNIT_HEAT_CONDUCTIVITY,             // 53
	UNIT_HEAT_CAPACITY,                 // 54 
	UNIT_HEAT_CONV_COEF,                // 55 

	UNIT_POWER_PER_LEN_GLOB,            // 56
	UNIT_POWER_PER_LEN,                 // 57

	UNIT_HUMIDITY_CONV,                 // 58
	UNIT_VISCOSITY,                     // 59
	UNIT_VOL_DISCHARGE,                 // 60
	UNIT_TOTAL_NROF_TYPES
};


enum { UNIT_TOTAL_NROF_SIMPLE = 41 };

static const double ACC_GRAVITY = 9.80655; //m/s2


class _ZUTL_CLASS  UNIT_Simple
{
public:
	enum UNIT_baseUnits Id; // force , lengt etc
	CString label;       // "N","kN","m","C" etc 
	double factor;       //  versus reference unit, example:
						 //  let  kN is refernce unit -> factor for N  is 0.001
	short Ipos;          //  position on list of all units in UNIT_manager

	//constructors:
	//use it as first during  intialization phase in UNIT_manager::ListOf 
	UNIT_Simple(int ibase = UNIT_BASE_UNDEF, const CString& labIn = _T("-"), double x = 1.0, int posIN = 0);

	//use it after intialization phase
	UNIT_Simple(int i); // basing on position in UNIT_manager::ListOf 
	UNIT_Simple(const CString& labIn); // basing on label
	UNIT_Simple(char* labIn); // basing on label

	~UNIT_Simple() {}

	void Init();
};


class _ZUTL_CLASS   UNIT_Set
	// this class describe used units system (reference,global,temporary)
	// i.e. specify which detailed unit is used for given base type such as 
	// force,len,anglr,time,temp
{
public:
	UNIT_Simple Base[UNIT_BASE_NROF_ALL];

public:
	//constructor:
	UNIT_Set();
	UNIT_Set(const UNIT_Simple& force, const UNIT_Simple& length, const UNIT_Simple& angle,
		const UNIT_Simple& time, const UNIT_Simple& temperature, 
		const UNIT_Simple& mass = UNIT_Simple("kg"), 
		const UNIT_Simple& energy = UNIT_Simple("J"), 
		const UNIT_Simple& power = UNIT_Simple("W"));

	UNIT_Set(UNIT_Simple& mass, UNIT_Simple& length);

	void Init();

	void SetBase(int i, UNIT_Simple& base);
	void SetBase(int i, CString& lab);
	void SetBase(CString& line);

	void ReadCfg(fstream& file);
	void WriteCfg(fstream& file);
	void Read(fstream& file);
	void Write(fstream& file);
	void WriteDAT(fstream& file);
	void GetLabel(CString& lab);

	int isUnitSystemDefined();
};



// this class describe a relation between  base units and complex unit
// for specified ZSOIL item (stress, verloc, volume etc..) identified by Id
class _ZUTL_CLASS UNIT_ComplexType
{
public:
	enum UNIT_ComplexUnits Id;
	enum UNIT_ComplexUnits _idGlob;
	int power[UNIT_BASE_NROF_ALL];

public:
	// constructors:
	UNIT_ComplexType(enum UNIT_ComplexUnits idIn = UNIT_DIMLESS, int pF = 0, int pL = 0, int pA = 0, int pt = 0, int pT = 0,
		int pM = 0, int pE = 0, int pP = 0, enum UNIT_ComplexUnits idGlobal = UNIT_DIMLESS);
	void Init();

	virtual double Multiplier(UNIT_Set& In, UNIT_Set& Out) const;

	virtual bool isDefinedInOtherUnitBase() { return Id != _idGlob;	}
	virtual enum UNIT_ComplexUnits GetGlobalUnitType() { return _idGlob; }

	CString& GetUnitSymbol(UNIT_Set& UnitSet, const vector<UNIT_baseUnits>& baseUnits = {});
};


class _ZUTL_CLASS UNIT_Cfg
{
public:
	CString Name;
	UNIT_Set Glo;
	UNIT_Set Output;

public:
	UNIT_Cfg();
	UNIT_Cfg(const UNIT_Cfg&) {}

	virtual int operator == (const UNIT_Cfg&);
	virtual UNIT_Cfg& operator = (const UNIT_Cfg& p);

	void ReadCfg(fstream& file);
	void WriteCfg(fstream& file);
};


class _ZUTL_CLASS UNIT_CfgBag : public Bag<UNIT_Cfg>
{
public:
	CString DefaultUnitName;

public:
	void ReadCfg(fstream& file);
	void WriteCfg(fstream& file);
	void WriteDefaultUnit(fstream& file);

	virtual UNIT_CfgBag& operator = (const UNIT_CfgBag& p);

	void GetUnits(CString uName, UNIT_Set& Input, UNIT_Set& Output);
	void SetUnits(CString uName, UNIT_Set& Input, UNIT_Set& Output);
	void SetDefaultUnits(CString uName, UNIT_Set& Input, UNIT_Set& Output);
	void GetSelectedUnits(UNIT_Set& Input, UNIT_Set& Output);
	void AddUnit(CString uName);
	void DeleteUnit(CString uName);
};


class _ZUTL_CLASS UNIT_manager
{
protected:
	static int startPathDefined;

public:
	CString StartPath;

public:
	static double UnitMultInitToAct[UNIT_TOTAL_NROF_TYPES];
	static double UnitMultPrevToAct[UNIT_TOTAL_NROF_TYPES];
	static double UnitMultFromTempToGlob[UNIT_TOTAL_NROF_TYPES];

public:
	UNIT_manager();

	// list of types of ZSOIL items 
	static class UNIT_ComplexType ListOfPatterns[UNIT_TOTAL_NROF_TYPES];
	//static class UNIT_ComplexType ListOfPatterns2[UNIT_TOTAL_NROF_TYPES2];
	static class UNIT_Set Reference; // this  data do not have to be changeed 

	UNIT_CfgBag UnitLst;

	UNIT_Set Glo; // this is to be set in the job
	UNIT_Set Tmp; // this is to be set temporarily dialog
	UNIT_Set Output;
	UNIT_Set SI;

	//returns ptr to static list of simple units for given type of base unit:
	UNIT_Simple* ArrayOfSimpleForBaseUnit(enum UNIT_baseUnits Ibase, int* NrOF);

	//returns available labels for given type of compound unit:
	CString& LabelsForType(enum UNIT_ComplexUnits Type);

	//set global unit system:
	void SetGlobal(UNIT_Set& GloIn);

	//set temporary unit system:
	void SetTemporary(UNIT_Set& TmpIn);
	void SetTemporary(CString& line);

	// return multiplier of compound unit between two unit systems In and Out:
	// ie number such that: 
	// valueExpressedIn[Out] = Multiplier*valueExpressedin[In]
	double Multiplier(enum UNIT_ComplexUnits Type, UNIT_Set& In, UNIT_Set& Out);

	double MultFromGlobToTemp(enum UNIT_ComplexUnits Type);
	double MultFromTempToGlob(enum UNIT_ComplexUnits Type);
	double MultFromGlobToOutput(enum UNIT_ComplexUnits Type);

	double MultFromSIToGlob(enum UNIT_ComplexUnits Type);
	double MultFromSIToOutput(enum UNIT_ComplexUnits Type);
	double MultFromGlobToSI(enum UNIT_ComplexUnits Type);
	double MultFromOutputToSI(enum UNIT_ComplexUnits Type);

	double ReCalcFormSIToGlob(enum UNIT_ComplexUnits Type, double val);
	double ReCalcFormSIToOutput(enum UNIT_ComplexUnits Type, double val);

	double ReCalcFromMassToGlob(enum UNIT_ComplexUnits Typ1, enum UNIT_ComplexUnits Typ2, double val);
	double FromOtherBaseToGlobalMult(enum UNIT_ComplexUnits typOtherBase, enum UNIT_ComplexUnits TypGlobalBase);
	double FromGlobalToOtherBaseMult(enum UNIT_ComplexUnits typOtherBase, enum UNIT_ComplexUnits TypGlobalBase);

	CString& GetLabel(enum UNIT_ComplexUnits uTyp, UNIT_Set& UnitSet);
	CString& GetLabel(enum UNIT_ComplexUnits uTyp, UNIT_Set& UnitSet, vector<UNIT_baseUnits> &baseUnits);

	CString& GetLabelForGlobalUnits(enum UNIT_ComplexUnits typ);
	CString& GetLabelForOutputUnits(enum UNIT_ComplexUnits typ);
	CString& GetLabelForTemporaryUnits(enum UNIT_ComplexUnits typ);

	void GetLabelForOutputUnits(CString& lab);
	void GetLabelForTemporaryUnits(CString& lab);

	int isUnitSystemDefined();

	void ReadCfg();
	void WriteCfg();

	void Read(fstream& file);
	void Write(fstream& file);

	void RecalcUnitMult();
	void RecalcUnitMultFromTempToGlob();
};


//==============================================
//    class UNIT_field
//==============================================
class _ZUTL_CLASS UNIT_field : public UNIT_fieldBase
{
public:
	enum eAngleUnit
	{
		NOT_ANGLE = -1,
		ANGLE_DEG,
		ANGLE_RAD,
	};

public:
	enum UNIT_ComplexUnits Typ;
	enum UNIT_ComplexUnits TypGlobal;
	CString Label;
	UNIT_manager* mngr;
	UNIT_Set _GloSet;
	vector<UNIT_baseUnits> _baseUnits;

protected:
	enum eAngleUnit angleType;

public:
	UNIT_field();
	UNIT_field(const UNIT_fieldBase& val);
	UNIT_field(enum UNIT_ComplexUnits typ);
	UNIT_field(enum UNIT_ComplexUnits typ, double val);
	virtual ~UNIT_field();

	void Init();
	virtual void operator = (double val);
	virtual UNIT_field& operator = (const UNIT_field& val);
	virtual UNIT_field& operator = (const UNIT_fieldBase& val);
	virtual bool operator == (const UNIT_field& val);

	_ZUTL_FUN friend fstream& operator>> (fstream& file, UNIT_field& dat);
	_ZUTL_FUN friend fstream& operator<< (fstream& file, UNIT_field& dat);

	void WriteAsDouble(fstream& file, int width);

	void SetUnitType(enum UNIT_ComplexUnits typ);
	void SetUnitType(enum UNIT_ComplexUnits typOther, enum UNIT_ComplexUnits typGlobal);

	void SetDouble(double valDouble, enum UNIT_ComplexUnits unitTyp, const CString& valName);
	void SetInt(int valInt, const CString& valName);
	void SetAngleType(enum eAngleUnit tp);

	virtual int  SetTempUnits();
	virtual void GetValueAfterReCalc();
	virtual void ReCalcFromTempToGlo();
	virtual void ReCalcFromTempToGloDirectly();
	virtual double GetValueFromOtherBaseToGlobal() const;
	virtual double GetValueFromGlobalBaseToOther() const;

	virtual void SetLabel(UNIT_Set& UnitSet);
	virtual UNIT_Set& GetGlobalUnits();

	CString& GetLabel();

	CString GetValueAsString();
	CString GetFormattedValueAsString(const CString& frmt);
	double GetRad();
};


//==============================================
//    class UNIT_Massfield
//==============================================

class _ZUTL_CLASS UNIT_Massfield : public UNIT_field
{
public:
	UNIT_Massfield();
	UNIT_Massfield(double val);

	virtual ~UNIT_Massfield();
	virtual void GetValueAfterReCalc();
	virtual void ReCalcFromTempToGlo() {}
	virtual void ReCalcFromTempToGloDirectly() {}

	virtual int SetTempUnits();

	virtual void operator = (double val) { value = val; }

	UNIT_Set& GetGlobalUnits() { return _GloSet; }
};



//==============================================
//    global exported functions
//==============================================

_ZUTL_FUN UNIT_manager* GetUnitManager();
_ZUTL_FUN int  SetGlobalUnits();
_ZUTL_FUN int  SetGlobalOutputUnits();

_ZUTL_FUN int SetTemporaryUnits();

_ZUTL_FUN void SetUndefinedUnits();
_ZUTL_FUN void ReadUnits(fstream& file);
_ZUTL_FUN void WriteUnits(fstream& file);
_ZUTL_FUN void WriteUnitsToDAT(fstream& file);

_ZUTL_FUN void ZUTL_SetUnits(CString& lineINPUT, CString& lineOUTPUT);
_ZUTL_FUN void ZUTL_SetUnits(CString& name, CString& lineINPUT, CString& lineOUTPUT);
_ZUTL_FUN void ZUTL_SetUnits(char* lineINPUT, char* lineOUTPUT);

_ZUTL_FUN double GetMultForNewUnits(enum UNIT_ComplexUnits typ);
_ZUTL_FUN double GetMultForOutputUnits(enum UNIT_ComplexUnits typ);

_ZUTL_FUN double GetMultFromSIToGlob(enum UNIT_ComplexUnits typ);
_ZUTL_FUN double GetMultFromSIToOutput(enum UNIT_ComplexUnits typ);

_ZUTL_FUN double GetMultFromGlobToSI(enum UNIT_ComplexUnits typ);
_ZUTL_FUN double GetMultFromOutputToSI(enum UNIT_ComplexUnits typ);

_ZUTL_FUN CString& ZUTL_GetUnitSymbolForGlobalUnits(enum UNIT_ComplexUnits typ);
_ZUTL_FUN CString& ZUTL_GetUnitSymbolForOutputUnits(enum UNIT_ComplexUnits typ);
_ZUTL_FUN CString& ZUTL_GetUnitSymbolForTemporaryUnits(enum UNIT_ComplexUnits typ);

_ZUTL_FUN void ZUTL_GetSymbolsForGlobalUnits(CString& lab);
_ZUTL_FUN void ZUTL_GetSymbolsForOutputUnits(CString& lab);
_ZUTL_FUN void ZUTL_GetSymbolsForTemporaryUnits(CString& lab);

_ZUTL_FUN int ZUTL_RecalcUnit(char* lineINPUT, int typ, double* value);

_ZUTL_FUN enum UNIT_ComplexUnits ZUTL_GetUnitTypeFromUnitLabel(CString& lab);

#endif /* UNIT.h */